import os
import sys
import subprocess
import platform
import shutil
from datetime import datetime

# ===================== CONFIG ===================== #

TEMP_VENV = ".anthro_tmp_env"
REQUIRED_PACKAGES = ["pillow"]

FONT_PATH = "Orbitron-VariableFont_wght.ttf"
BRANDING_IMAGE = "branding.png"
OUTPUT_DIR = "output"

BACKGROUND_COLOR = (0, 0, 0)
TEXT_COLOR = (255, 255, 255)

BRANDING_WIDTH_RATIO = 0.35
LINE_SPACING = 1.3  # Adds breathing room between lines to prevent clipping

# DYNAMIC PADDING (Adjusts based on image size)
BUFFER_RATIO = 0.02 # 2% of image height for gaps

# ================================================== #
# ---------------- VENV BOOTSTRAP ------------------ #

def is_venv():
    return sys.prefix != sys.base_prefix

def venv_python():
    if platform.system() == "Windows":
        return os.path.join(TEMP_VENV, "Scripts", "python.exe")
    return os.path.join(TEMP_VENV, "bin", "python")

def ensure_env():
    if not os.path.exists(TEMP_VENV):
        print("Creating environment...")
        subprocess.check_call([sys.executable, "-m", "venv", TEMP_VENV])
        py = venv_python()
        subprocess.check_call([py, "-m", "pip", "install", "--upgrade", "pip"])
        subprocess.check_call([py, "-m", "pip", "install", *REQUIRED_PACKAGES])

# ---------------- STAMP TEXT ---------------------- #

def load_stamp_lines(txt_path):
    timestamp = datetime.utcnow().strftime("%Y-%m-%d %H:%M:%S UTC")
    lines = []
    if not os.path.exists(txt_path):
        return ["FILE NOT FOUND: " + txt_path]

    with open(txt_path, "r", encoding="utf-8") as f:
        for line in f:
            line = line.strip()
            if not line: continue
            if "[TIMESTAMP]" in line:
                line = line.replace("[TIMESTAMP]", timestamp)
            lines.append(line)
    return lines

# ---------------- FONT FITTING -------------------- #

def compute_best_font(draw, font_path, lines, max_width):
    from PIL import ImageFont
    
    # Optimization: Binary search for font size
    low = 10
    high = max_width // 2
    best_size = low
    
    target_width = max_width * 0.90 # Leave 5% margin on each side

    while low <= high:
        mid = (low + high) // 2
        font = ImageFont.truetype(font_path, mid)
        widest = max(draw.textlength(line, font=font) for line in lines)
        
        if widest <= target_width:
            best_size = mid
            low = mid + 1
        else:
            high = mid - 1
            
    return ImageFont.truetype(font_path, best_size)

# ---------------- IMAGE PROCESSING ---------------- #

def stamp_images(txt_path):
    from PIL import Image, ImageDraw

    if not os.path.exists(FONT_PATH):
        raise FileNotFoundError(f"Font missing: {FONT_PATH}")
    if not os.path.exists(BRANDING_IMAGE):
        raise FileNotFoundError(f"Branding image missing: {BRANDING_IMAGE}")

    stamp_lines = load_stamp_lines(txt_path)
    os.makedirs(OUTPUT_DIR, exist_ok=True)
    branding = Image.open(BRANDING_IMAGE).convert("RGBA")

    for filename in os.listdir("."):
        if filename.lower().endswith(SUPPORTED_EXTENSIONS):
            src = os.path.abspath(filename)
            dst = os.path.join(OUTPUT_DIR, filename)

            if os.path.exists(dst):
                print(f"⚠ Skipping: {filename}")
                continue

            with Image.open(src) as img:
                img = img.convert("RGBA")
                w, h = img.size
                
                # Setup drawing context for font measurement
                draw_tmp = ImageDraw.Draw(img)
                font = compute_best_font(draw_tmp, FONT_PATH, stamp_lines, w)
                
                # Calculate Line Heights
                ascent, descent = font.getmetrics()
                line_h = ascent + descent
                total_text_h = int(len(stamp_lines) * line_h * LINE_SPACING)

                # Branding scaling
                brand_w = int(w * BRANDING_WIDTH_RATIO)
                brand_h = int(branding.height * (brand_w / branding.width))
                branding_resized = branding.resize((brand_w, brand_h), Image.LANCZOS)

                # Dynamic Padding
                pad = int(h * BUFFER_RATIO)
                if pad < 20: pad = 20 # Minimum safety floor

                # TOTAL CANVAS HEIGHT
                extra_h = pad + brand_h + pad + total_text_h + pad
                
                canvas = Image.new("RGBA", (w, h + extra_h), BACKGROUND_COLOR)
                canvas.paste(img, (0, 0))
                draw = ImageDraw.Draw(canvas)

                # Draw Branding
                y_cursor = h + pad
                brand_x = (w - brand_w) // 2
                canvas.paste(branding_resized, (brand_x, y_cursor), branding_resized)

                # Draw Text
                y_cursor += brand_h + pad
                for line in stamp_lines:
                    text_w = draw.textlength(line, font=font)
                    draw.text(((w - text_w) // 2, y_cursor), line, fill=TEXT_COLOR, font=font)
                    y_cursor += int(line_h * LINE_SPACING)

                canvas.convert("RGB").save(dst, quality=95, subsampling=0)
                print(f"✔ Processed: {filename}")

SUPPORTED_EXTENSIONS = (".png", ".jpg", ".jpeg", ".webp", ".tiff", ".bmp")

if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: python STAMP_IMAGES.py stamp.txt")
        sys.exit(1)

    if not is_venv():
        ensure_env()
        py = venv_python()
        subprocess.check_call([py, os.path.abspath(__file__), sys.argv[1]])
        sys.exit(0)

    stamp_images(sys.argv[1])